import { NextFunction, Request, Response } from "express";
import formidable, { Fields, Files } from "formidable";
import { TypedRequestAuth } from "./auth.controller";
import { asyncWrap } from "../helpers";
import { postDb, userDb } from "../db";
import { TypedRequest } from "../types";

export const createFileAndPost = asyncWrap(async (reqExp, res, next) => {
  const req = reqExp as TypedRequestAuth<{ userId: string }>;

  const { userId } = req.auth;

  const form = formidable({
    uploadDir: "uploads",
    keepExtensions: true,
    multiples: true,
  });

  const fileIdArr = new Array();

  form.parse(req, async (err, fields, files) => {
    if (!Array.isArray(fields.title)) {
      const title = fields.title;
      if (!Array.isArray(fields.text)) {
        const text = fields.text;
        if (!Array.isArray(fields.theme)) {
          const theme = fields.theme;
          if (!Array.isArray(fields.city)) {
            const city = fields.city;

            if (Array.isArray(files.picture)) {
              for (var i = 0; i < files.picture.length; i++) {
                const originalfilename = files.picture?.[i].originalFilename;
                const newfilename = files.picture?.[i].newFilename;
                const filepath = files.picture?.[i].filepath;

                const filesRes = await postDb.createFilesRow(
                  originalfilename,
                  newfilename,
                  filepath
                );

                fileIdArr.push(filesRes);
              }
            } else if (!Array.isArray(files.picture)) {
              const originalfilename = files.picture.originalFilename;
              const newfilename = files.picture.newFilename;
              const filepath = files.picture.filepath;

              const filesRes = await postDb.createFilesRow(
                originalfilename,
                newfilename,
                filepath
              );

              fileIdArr.push(filesRes);
            } // file or one or more

            const postRes = await postDb.createPostRow({
              title,
              text,
              theme,
              city,
              date: Date.now(),
              counts: 0,
              user: userId,
              file: fileIdArr,
            });
          }
        }
      }
    }
  });
});

export const getAllPost = asyncWrap(async (req, res) => {
  const posts = await postDb.getPosts();
  // console.log(posts);

  return res.json(posts);
});

export const getFiles = asyncWrap(async (req, res) => {
  const { postId } = req.params;
  // console.log("나는 말하는 고구마.", postId);
  const files = await postDb.getFilesByPostId(postId);

  return res.json(files);
});

export const addCounts = asyncWrap(async (req, res) => {
  const { postId } = req.params;
  const { counts } = req.body as {
    counts: number;
  };
  // console.log(postId, counts);

  const updateCounts = await postDb.addOneCount(postId, counts);

  return res.json(updateCounts);
});

export const userByPostId = (
  reqExp: Request,
  res: Response,
  next: NextFunction,
  postId: string
) => {
  const req = reqExp as TypedRequest;
  req.user = userDb.findUserByPostId(postId);
  next();
};

export const getOnePost = asyncWrap(async (req, res) => {
  const { postId } = req.params;
  const post = await postDb.getPost(postId);

  return res.json(post);
});

export const deleteOnePost = asyncWrap(async (req, res) => {
  const { postId } = req.params;
  // console.log(postId);
  const deleteCount = await postDb.deletePost(postId);

  return res.json(deleteCount);
});

export const updatePost = asyncWrap(async (reqExp, res) => {
  const req = reqExp as TypedRequestAuth<{ userId: string }>;

  const { title, text, theme, city, date } = req.body as {
    title: string;
    text: string;
    theme: string;
    city: string;
    date: Date;
    counts: number;
  };

  const userId = req.auth.userId;
  const { postId } = req.params;

  const updatePost = await postDb.updateOnePost(
    {
      title,
      text,
      theme,
      city,
      date: Date.now(),
      counts: req.body.counts,
      user: userId,
    },
    postId
  );

  // console.log("게시글 수정 후", updatePost);

  return res.json(updatePost);
});
