import { KU } from "../db/index.js";
import sequelize from 'sequelize';

const { Op } = sequelize

const findbyId = async (req, res, next) => {
    try {
        const id = req.scheduleId
        if (id) {
            const findSchedule = await KU.findOne({ where: { id: id } })
            if (!findSchedule) throw new Error("해당 일정을 찾지 못했습니다.")
            else {
                const { title, start, end, memo } = findSchedule
                const startDate = dateToString(start, "full")
                const endDate = dateToString(end, "full")
                req.schedule = { title, startDate: startDate, endDate: endDate, memo }
            }
            next()
        } else next()
    } catch (error) {
        return res.status(500).send(error.message || "일정 가져오는 중 에러 발생")
    }
}

const findbyDate = async (req, res, next) => {
    try {
        if (req.date) {
            const date = new Date(req.date)
            const findList = await KU.findAll({
                where: {
                    [Op.and]: [
                        {
                            start: {
                                [Op.lte]: date
                            }
                        }, {
                            end: {
                                [Op.gte]: date
                            }
                        }
                    ]
                },
                order: [['updatedAt', 'DESC']]
            })
            findList.forEach(schedule => {
                schedule.dataValues.start = dateToString(schedule.start, "twoYear")
                schedule.dataValues.end = dateToString(schedule.end, "twoYear")
            })
            req.scheduleList = findList
            next()
        } else next()
    } catch (error) {
        return res.status(500).send(error.message || "일정 가져오는 중 에러 발생")
    }
}

const create = async (req, res) => {
    try {
        const { title, startDate, endDate, memo } = req.body
        const start = new Date(startDate)
        const end = new Date(endDate)
        const newSchedule = await KU.create({ title: title, start: start, end: end, memo: memo })
        return res.json(newSchedule)
    } catch (error) {
        return res.status(500).send(error.message || "일정 등록 중 에러 발생")
    }
}

const edit = async (req, res) => {
    try {
        const { scheduleId } = req.query
        const { title, startDate, endDate, memo } = req.body
        const updated = await KU.update({ title: title, start: startDate, end: endDate, memo: memo }, { where: { id: scheduleId } })
        if (!updated) throw new Error("해당 일정의 일부 정보를 수정하는데 실패하였습니다.")
        else return res.send(200)
    } catch (error) {
        return res.status(500).send(error.message || "일정 수정 중 에러 발생")
    }
}

const remove = async (req, res) => {
    try {
        const { scheduleId } = req.query
        const deleted = await KU.destroy({ where: { id: scheduleId } })
        if (!deleted) throw new Error("해당 일정을 삭제하는데 실패하였습니다.")
        else return res.send(200)
    } catch (error) {
        return res.status(500).send(error.message || "일정 삭제 중 에러 발생")
    }
}

const querySeparation = async (req, res, next) => {
    try {
        const { scheduleId, date } = req.query
        req.scheduleId = scheduleId
        req.date = date
        next()
    } catch (error) {
        return res.status(500).send(error.message || "일정 가져오는 중 에러 발생")
    }
}

const send = async (req, res) => {
    try {
        const result = req.schedule || req.scheduleList
        return res.json(result)
    } catch (error) {
        return res.status(500).send(error.message || "일정 가져오는 중 에러 발생")
    }
}

function dateToString(dateObj, method) {
    const year = dateObj.getFullYear()
    const year_disit = String(year).substring(2, 4)
    const month = dateObj.getMonth() + 1
    const date = dateObj.getDate()

    if (method === "full") return [year, (month > 9 ? "" : "0") + month, (date > 9 ? "" : "0") + date].join("-")
    else if (method === "twoYear") return [year_disit, (month > 9 ? "" : "0") + month, (date > 9 ? "" : "0") + date].join("-")
}

export default {
    findbyId,
    findbyDate,
    create,
    edit,
    remove,
    querySeparation,
    send
}