import { useState, useEffect, useRef } from "react";
import { useHistory, useParams } from "react-router-dom";
import Date from "./DateSet";
import moment from 'moment';
import FullCalendar, { createPlugin } from '@fullcalendar/react';
import interactionPlugin from "@fullcalendar/interaction";
import bootstrapPlugin from '@fullcalendar/bootstrap';

const Weekly = () => {
    const history = useHistory()
    const { date } = useParams()
    const [chooseDate, setChooseDate] = useState(moment(date).format("YYYY-MM-DD"))
    const [week, setWeek] = useState([
        { date: moment(date).day(0).format("YYYY-MM-DD"), rate: "75" },
        { date: moment(date).day(1).format("YYYY-MM-DD"), rate: "85" },
        { date: moment(date).day(2).format("YYYY-MM-DD"), rate: "40" },
        { date: moment(date).day(3).format("YYYY-MM-DD"), rate: "100" },
        { date: moment(date).day(4).format("YYYY-MM-DD"), rate: "" },
        { date: moment(date).day(5).format("YYYY-MM-DD"), rate: "0" },
        { date: moment(date).day(6).format("YYYY-MM-DD"), rate: "" }
    ])
    const calendarRef = useRef(null)
    let calendar = null

    useEffect(() => {
        if (calendarRef && calendarRef.current) {
            calendar = calendarRef.current.getApi()
        }
    })

    useEffect(() => {
        setChooseDate(moment(date).format("YYYY-MM-DD"))
        setWeek([
            { date: moment(date).day(0).format("YYYY-MM-DD"), rate: "7" },
            { date: moment(date).day(1).format("YYYY-MM-DD"), rate: "8" },
            { date: moment(date).day(2).format("YYYY-MM-DD"), rate: "4" },
            { date: moment(date).day(3).format("YYYY-MM-DD"), rate: "100" },
            { date: moment(date).day(4).format("YYYY-MM-DD"), rate: "" },
            { date: moment(date).day(5).format("YYYY-MM-DD"), rate: "0" },
            { date: moment(date).day(6).format("YYYY-MM-DD"), rate: "" }
        ])
    }, [date])

    function prev() {
        calendar.prev()
        let date = moment(calendar.getDate()).format('YYYY-MM-DD')
        history.push(`/todo/${date}`)
    }

    function next() {
        calendar.next()
        let date = moment(calendar.getDate()).format('YYYY-MM-DD')
        history.push(`/todo/${date}`)
    }

    function gotoDate(date) {
        calendar.gotoDate(date)
        history.push(`/todo/${date}`)
    }

    const CustomeWeeklyView = ({ dateProfile }) => {
        let current = moment(dateProfile.currentRange.start).format("YYYY-MM-DD")
        
        return (
            <div className="fc-custom-view weekly-view d-flex row-cols-9" style={{ cursor: "pointer" }}>
                <i className="col bi bi-chevron-left align-self-center" onClick={prev} style={{ fontSize: "2em" }} />
                {week.map((info, idx) => <Date index={idx} info={info} today={moment(info.date).isSame(current) ? true : false} handleClick={gotoDate} />)}
                <i className="col bi bi-chevron-right align-self-center" onClick={next} style={{ fontSize: "2em" }} />
            </div>
        )
    }

    const customViewPlugin = createPlugin({
        views: {
            week: CustomeWeeklyView
        }
    })

    return (
        <FullCalendar
            ref={calendarRef}
            plugins={[customViewPlugin, interactionPlugin, bootstrapPlugin]}
            initialView="week"
            initialDate={chooseDate}
            headerToolbar={{
                start: 'title',
                center: '',
                end: ''
            }}
            titleFormat={() => moment(chooseDate).format("YYYY[년 ]MM[월]")}
            views={{
                week: {
                    validRange: {
                        start: moment(chooseDate).subtract(3, 'years').format('YYYY-MM[-01]'),
                        end: moment(chooseDate).add(3, 'years').add(1, 'months').format('YYYY-MM[-01]')
                    },
                    dateIncrement: {
                        days: 7
                    }
                }
            }}
            timeZone="local"
            themeSystem='bootstrap'
            height="fit-content"
        />
    )
}

export default Weekly