import formidable from 'formidable'
import fs from 'fs'
import dbErrorHandler from '../helpers/dbErrorHandler.js'
import Problem from './problem.model.js'
import Quiz from './quiz.model.js'
import extend from 'lodash/extend.js'

const create = async (req, res) => {
  try {
    const { title, problems, startAt, endAt, course } = req.body

    const quiz = new Quiz()
    // console.log('quiz in quiz.controller:', quiz);

    for await (let problem of problems) {
      // console.log('problem in quiz.controller:', problem);
      const p = new Problem(problem)
      // console.log('problem in quiz.controller:', p);
      p.author = req.profile
      p.quiz = quiz._id
      await p.save()
      quiz.problems.push(p._id)
    }

    quiz.title = title
    quiz.startAt = startAt
    quiz.endAt = endAt
    quiz.course = course
    quiz.author = req.profile

    // console.log('quiz in quiz.controller:', quiz);
    await quiz.save()
    quiz.author.hashedPassword = undefined
    quiz.author.salt = undefined
    res.json(quiz)
  } catch (error) {
    return res.status(400).json({
      error: 'Quiz save DB error' + error
    })
  }
}

const createProblem = async (req, res) => {
  try {
    const problem = new Problem(req.body)
    problem.author = req.profile._id

    await problem.save()

    // push the problem to the quiz
    if (problem.quiz) {
      const quiz = await Quiz.findById(problem.quiz)
      quiz.problems.push(problem)
      await quiz.save()
    }
    res.json(problem)
  } catch (error) {
    return res.status(400).json({
      error: dbErrorHandler.getErrorMessage(error)
    })
  }
}

const isAuthor = (req, res, next) => {
  const isAuthor = req.auth && req.quiz && req.auth._id == req.quiz.author._id
  if (!isAuthor) {
    return res.status(403).json({
      error: 'User is not an author'
    })
  }
  next()
}

const isProblemAuthor = (req, res, next) => {
  const isProblemAuthor = req.auth && req.problem && req.auth._id == req.problem.author._id
  if (!isProblemAuthor) {
    return res.status(403).json({
      error: 'User is not an author of the problem'
    })
  }
  next()
}

const read = async (req, res) => {
  let quiz = req.quiz
  res.json(quiz)
}

const readProblem = async (req, res) => {
  let problem = req.problem
  res.json(problem)
}

const updateProblem = async (req, res) => {
  try {
    let problem = req.problem
    console.log('req.body in updateProblem in quiz.controller', req.body);
    problem = extend(problem, req.body)
    // problem.question = req.body.question
    // problem.answers = req.body.answers
    problem.updated = new Date()
    console.log('updated problem in updateProblem in quiz.controller', problem);
    await problem.save()
    res.json(problem)
  } catch (error) {
    return res.status(400).json({
      error: dbErrorHandler.getErrorMessage(error)
    })
  }
}

const removeProblem = async (req, res) => {
  try {
    const problem = req.problem
    await Quiz.findByIdAndUpdate(problem.quiz, {
      $pull: {problems: problem._id}
    })
    const deletedProblem = await problem.remove()
    res.json(deletedProblem)      
  } catch (error) {
    return res.status(400).json({
      error: dbErrorHandler.getErrorMessage(error)
    })
  }
}

const listByUserId = async (req, res) => {
  try {
    const authorId = req.profile._id
    const quizzes = await Quiz.find({author: authorId}).exec()
    // console.log('quizzes in listByUserId:', quizzes);
    res.json(quizzes)      
  } catch (error) {
    return res.status(400).json({
      error: dbErrorHandler.getErrorMessage(error)
    })
  }
}

const quizById = async (req, res, next, id) => {
  try {
    const quiz = await Quiz.findById(id)
      .populate('author', '_id name')
      .populate('problems')
      .populate('course', '_id name')
      .exec()
    if (!quiz) {
      return res.status(400).json({
        error: 'Quiz not found'
      })
    }
    req.quiz = quiz
    next()
  } catch (error) {
    return res.status(400).json({
      error: 'Quiz by id query db error: ' + error
    })
  }
}

const problemById = async (req, res, next, id) => {
  try {
    const problem = await Problem.findById(id)
      .populate('author', '_id name')
      .exec()
    if (!problem) {
      return res.status(400).json({
        error: 'Problem not found'
      })
    }
    req.problem = problem
    next()
  } catch (error) {
    return res.status(400).json({
      error: dbErrorHandler.getErrorMessage(error)
    })
  }
}

export default {
  create,
  createProblem,
  read,
  readProblem,
  updateProblem,
  removeProblem,
  isAuthor,
  isProblemAuthor,
  listByUserId,
  quizById,
  problemById,
}