import User from "../models/User.js"
import isLength from 'validator/lib/isLength.js'
import isEmail from 'validator/lib/isEmail.js'
import bcrypt from "bcryptjs";
import multer from "multer";

const upload = multer({ dest: 'uploads/' })
const profileUpload = upload.fields([
    { name: 'imageUrl', maxCount: 1 },
])

const signup = async (req, res) => {

    const { username, nickname, email, password } = req.body

    try {
        if (!isLength(username, { min: 3, max: 10 })) {
            return res.status(422).send('이름은 3-10자 사이입니다')
        } else if (!isLength(nickname, { min: 2, max: 10 })) {
            return res.status(422).send('별명은 2-10자 사이입니다.')
        } else if (!isLength(password, { min: 6 })) {
            return res.status(422).send('비밀번호는 6자 이상입니다.')
        } else if (!isEmail(email)) {
            return res.status(422).send('유효하지 않은 이메일 형식입니다')
        }

        const user = await User.findOne({ email })
        if (user) {
            return res.status(422).send(`${email}이 이미 사용중입니다.`)
        }

        const hash = await bcrypt.hash(password, 10)

        const newUser = await new User({
            username,
            nickname,
            email,
            password: hash,
        }).save()
        res.json(newUser)
    } catch (error) {
        //알수없는 모든 에러발생시 처리
        console.log(error)
        res.status(500).send('회원가입 에러발생하였습니다.')
    }
}

const update = async (req, res) => {
    const username = req.user.username
    const newNickname = req.body.newNickname
    try {
        await User.updateOne({ 'username': username }, { 'nickname': newNickname })
        let imageUrl = ''
        if (req.files['imageUrl']) {
            imageUrl = req.files['imageUrl'][0]
        }
        const user = req.user
        if (req.files['imageUrl']) {
            user.profileimg = imageUrl.filename //우리가 사용할 uesr.profileimg에다가 imageUrl의 filename을 저장
        }
        const updatedUser = await user.save()
        res.json(updatedUser)
    } catch (error) {
        console.log(error)
        res.status(500).send('프로필 업데이트 실패')
    }
}

const userById = async (req, res, next, id) => {    //id로 찾아와서 req.user를 우리가 찾는 user로 저장
    try {
        const user = await User.findById(id)
        if (!user) {
            res.status(404).send('사용자를 찾을 수 없습니다.')
        }
        req.user = user
        next()
    } catch (error) {
        console.log(error)
        res.status(500).send('사용자 아이디 검색 실패')
    }
}
const getProfile = (req, res) => {
    res.json(req.user)
}

const checkUser = async (req, res) => {
    try {
        const info = await User.findOne({ _id: req.query._id })
        return res.json(info)
    } catch (error) {
        console.log(error)
        res.status(500).send('사용자 검색 실패')
    }
}

const loginNavbar = async (req, res) => {
    try {
        let user = await User.findOne(req.body).select('username').exec()
        return res.json(user)
    } catch (error) {
        alert('올바르지 못한 접근입니다.')
    }
}

export default { signup, profileUpload, update, userById, getProfile, loginNavbar, checkUser }
// {} : 객체로 return함
